//
//  ContentView.swift
//  Headlines WatchKit Extension
//
//  Created by Zac White on 9/24/19.
//  Copyright © 2019 Zac White. All rights reserved.
//

import SwiftUI
import UIKit.UIImage

struct HeaderImage: View {
    let uiImage: UIImage

    private var image: some View {
        Image(uiImage: uiImage)
            .resizable()
            .aspectRatio(contentMode: .fill)
    }

    var body: some View {
        Rectangle()
            .fill(Color.clear)
            .overlay(image, alignment: .top)
            .overlay(
                LinearGradient(
                    gradient: Gradient(
                        colors: [.black, .clear, .clear]
                    ),
                    startPoint: .bottom,
                    endPoint: .top)
            )
            .cornerRadius(5)
            .clipped()
            .compositingGroup()
    }
}

struct ArticleView: View {

    let article: Article

    @State private var offset: CGPoint = .zero

    private enum Constants {
        static let topSpace: CGFloat = 100
    }

    var body: some View {
        GeometryReader { proxy in
            ZStack(alignment: .top) {
                HeaderImage(uiImage: self.article.image)
                    .frame(height: Constants.topSpace * 1.5)
                    .opacity(Double(max((Constants.topSpace + self.offset.y) / Constants.topSpace, 0)))
                    .offset(x: 0, y: self.offset.y > 0 ? self.offset.y : self.offset.y / 2)

                OffsetScrollView(offset: self.$offset) {
                    VStack(alignment: .leading, spacing: 5) {
                        Rectangle()
                            .frame(width: 0, height: Constants.topSpace)
                        Text(self.article.headline)
                            .font(.headline)
                            .fixedSize(horizontal: false, vertical: true)
                        Text(self.article.body)
                            .font(.body)
                            .fixedSize(horizontal: false, vertical: true)
                    }
                    .padding()
                }
                .edgesIgnoringSafeArea(.all)
            }
        }
    }
}

struct ContentView_Previews: PreviewProvider {
    static var previews: some View {
        ArticleView(article: .testCorgiArticle)
    }
}
